# BLE Central Proximity Sensory Example Walkthrough

## Introduction

In this tutorial, the ble_proximity_sensor central example code for the espressif chipsets with BLE5.0 support is reviewed. This example creates GATT client and performs passive scan, it then connects to peripheral device if the device advertises connectability and the device advertises support for the Link Loss service (0x1803) as primary service UUID. After connection it enables bonding and link encryprion if the `Enable Link Encryption` flag is set in the example config.


## Includes

This example is located in the examples folder of the ESP-IDF under the [ble_proximity_sensor/proximity_sensor_cent/main](../main). The [main.c](../main/main.c) file located in the main folder contains all the functionality that we are going to review. The header files contained in [main.c](../main/main.c) are:
```c
#include "esp_log.h"
#include "nvs_flash.h"

/* BLE */
#include "nimble/nimble_port.h"
#include "nimble/nimble_port_freertos.h"
#include "host/ble_hs.h"
#include "host/util/util.h"
#include "console/console.h"
#include "services/gap/ble_svc_gap.h"
#include "ble_prox_cent.h"
```

These `includes` are required for the FreeRTOS and underlying system components to run, including the logging functionality and a library to store data in non-volatile flash memory. We are interested in `“nimble_port.h”`, `“nimble_port_freertos.h”`, `"ble_hs.h"` and `“ble_svc_gap.h”`, `“ble_prox_cent.h”` which expose the BLE APIs required to implement this example.

* `nimble_port.h`: Includes the declaration of functions required for the initialization of the nimble stack.
* `nimble_port_freertos.h`: Initializes and enables nimble host task.
* `ble_hs.h`: Defines the functionalities to handle the host event.
* `ble_svc_gap.h`:Defines the macros for device name ,device apperance and declare the function to set them.
* `ble_prox_cent.h`: Defines the 16 bit Link Loss SVC UUID.


## Main Entry Point

The program’s entry point is the app_main() function:

```c
void
app_main(void)
{
    int rc;
    /* Initialize NVS — it is used to store PHY calibration data */
    esp_err_t ret = nvs_flash_init();
    if  (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    nimble_port_init();
    /* Configure the host. */
    ble_hs_cfg.reset_cb = ble_prox_cent_on_reset;
    ble_hs_cfg.sync_cb = ble_prox_cent_on_sync;
    ble_hs_cfg.store_status_cb = ble_store_util_status_rr;

    /* Initialize data structures to track connected peers. */
    rc = peer_init(MYNEWT_VAL(BLE_MAX_CONNECTIONS), 64, 64, 64);
    assert(rc == 0);

    /* Set the default device name. */
    rc = ble_svc_gap_device_name_set("nimble-prox-cent");
    assert(rc == 0);

    /* XXX Need to have template for store */
    ble_store_config_init();

    nimble_port_freertos_init(ble_prox_cent_host_task);

}
```

The main function starts by initializing the non-volatile storage library. This library allows to save the key-value pairs in flash memory.`nvs_flash_init()` stores the PHY calibration data. In a Bluetooth Low Energy (BLE) device, cryptographic keys used for encryption and authentication are often stored in Non-Volatile Storage (NVS).BLE stores the peer keys, CCCD keys, peer records, etc on NVS.By storing these keys in NVS, the BLE device can quickly retrieve them when needed, without the need for time-consuming key generations.
```c
esp_err_t ret = nvs_flash_init();
if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
    ESP_ERROR_CHECK(nvs_flash_erase());
    ret = nvs_flash_init();
}
ESP_ERROR_CHECK( ret );
```

## BT Controller and Stack Initialization

The main function calls `nimble_port_init()` to initialize BT Controller and nimble stack. This function initializes the BT controller by first creating its configuration structure named `esp_bt_controller_config_t` with default settings generated by the `BT_CONTROLLER_INIT_CONFIG_DEFAULT()` macro. It implements the Host Controller Interface (HCI) on the controller side, the Link Layer (LL), and the Physical Layer (PHY). The BT Controller is invisible to the user applications and deals with the lower layers of the BLE stack. The controller configuration includes setting the BT controller stack size, priority, and HCI baud rate. With the settings created, the BT controller is initialized and enabled with the `esp_bt_controller_init()` and `esp_bt_controller_enable()` functions:

```c
esp_bt_controller_config_t config_opts = BT_CONTROLLER_INIT_CONFIG_DEFAULT();
ret = esp_bt_controller_init(&config_opts);
```

Next, the controller is enabled in BLE Mode.

```c
ret = esp_bt_controller_enable(ESP_BT_MODE_BLE);
```
>The controller should be enabled in `ESP_BT_MODE_BLE` if you want to use the BLE mode.

There are four Bluetooth modes supported:

1. `ESP_BT_MODE_IDLE`: Bluetooth not running
2. `ESP_BT_MODE_BLE`: BLE mode
3. `ESP_BT_MODE_CLASSIC_BT`: BT Classic mode
4. `ESP_BT_MODE_BTDM`: Dual mode (BLE + BT Classic)

After the initialization of the BT controller, the nimble stack, which includes the common definitions and APIs for BLE, is initialized by using `esp_nimble_init()`:

```c
esp_err_t esp_nimble_init(void)
{

#if !SOC_ESP_NIMBLE_CONTROLLER
    /* Initialize the function pointers for OS porting */
    npl_freertos_funcs_init();

    npl_freertos_mempool_init();

    if(esp_nimble_hci_init() != ESP_OK) {
        ESP_LOGE(NIMBLE_PORT_LOG_TAG, "hci inits failed\n");
        return ESP_FAIL;
    }

    /* Initialize default event queue */
    ble_npl_eventq_init(&g_eventq_dflt);
    /* Initialize the global memory pool */
    os_mempool_module_init();
    os_msys_init();

#endif
    /* Initialize the host */
    ble_transport_hs_init();

    return ESP_OK;
}
```

The host is configured by setting up the callbacks for Stack-reset, Stack-sync, and Storage status

```c
 ble_hs_cfg.reset_cb = ble_prox_cent_on_reset;
 ble_hs_cfg.sync_cb = ble_prox_cent_on_sync;
 ble_hs_cfg.store_status_cb = ble_store_util_status_rr;
```

Further Data Structures are created and initialized to track connected peers using `peer_init()`. This function creates memory buffers to generate the memory pools like `peer_pool`, `peer_svc_pool`, `peer_chr_pool`, and `peer_dsc_pool`.
```c
 rc = peer_init(MYNEWT_VAL(BLE_MAX_CONNECTIONS), 64, 64, 64);
```

## Structure of Peer

The structure of a peer includes fields such as its connection handle, a pointer to the next peer, a list of discovered gatt services, tracking parameters for the service discovery process, and the callbacks that get executed when service discovery completes.

```c
struct peer {
    SLIST_ENTRY(peer) next;

    uint16_t conn_handle;


    struct peer_svc_list svcs;


    uint16_t disc_prev_chr_val;
    struct peer_svc *cur_svc;

    peer_disc_fn *disc_cb;
    void *disc_cb_arg;
};
```

The main function calls `ble_svc_gap_device_name_set()` to set the default device name. 'nimble-prox-cent' is passed as the default device name to this function.
```c
rc = ble_svc_gap_device_name_set("nimble-prox-cent");
```

main function calls  `ble_store_config_init()` to configure the host by setting up the storage callbacks which handle the read, write, and deletion of security material.
```c
/* XXX Need to have a template for store */
    ble_store_config_init();
```

The main function ends by creating a task where nimble will run using `nimble_port_freertos_init()`. This enables the nimble stack by using `esp_nimble_enable()`.
```c
nimble_port_freertos_init(ble_prox_cent_host_task);
```
`esp_nimble_enable()` create a task where the nimble host will run. It is not strictly necessary to have a separate task for the nimble host, but since something needs to handle the default queue, it is easier to create a separate task.


## Overall code flow.

It performs following GATT operations against the specified peer:

* Writes the alert level characteristic of link loss service.

* After the write operation is completed, reads the tx power level characteristic.

* It continuously calculated the path loss. If the path loss exceeds high threshold, it writes to alert level characteristic of immediate alert service of peripheral to start alerting.

* If the path loss drops below the low threshold, it writes to alert level characteristic of immediate alert service of peripheral to stop alerting.


## Write Operation

`ble_prox_cent_read_write_subscribe` stores the value handle of alert level characteristic of immediate alert service and then writes on the alert level characteristic of link loss service. If the peer does not support a required service, characteristic, or descriptor OR if a GATT procedure fails , then this function immediately terminates the connection.

```c
static void
ble_prox_cent_read_write_subscribe(const struct peer *peer)
{
    const struct peer_chr *chr;
    int rc;
    
    /* Storing the val handle of immediate alert characteristic */
    chr = peer_chr_find_uuid(peer,
                          BLE_UUID16_DECLARE(BLE_SVC_IMMEDIATE_ALERT_UUID16),
                          BLE_UUID16_DECLARE(BLE_SVC_PROX_CHR_UUID16_ALERT_LVL));
    if (chr != NULL) {
        conn_peer[peer->conn_handle].val_handle = chr->chr.val_handle;
    } else {
        MODLOG_DFLT(ERROR, "Error: Peer doesn't support the alert level"
                    " characteristic of immediate alert loss service\n");
    }
    
    /* Write alert level characteristic. */
    chr = peer_chr_find_uuid(peer,
                          BLE_UUID16_DECLARE(BLE_SVC_LINK_LOSS_UUID16),
                          BLE_UUID16_DECLARE(BLE_SVC_PROX_CHR_UUID16_ALERT_LVL));
    if (chr == NULL) {
        MODLOG_DFLT(ERROR, "Error: Peer doesn't support the alert level"
                    " characteristic\n");
        goto err;
    }
    
    rc = ble_gattc_write_flat(peer->conn_handle, chr->chr.val_handle,
                              &link_supervision_timeout, sizeof(link_supervision_timeout),
                              ble_prox_cent_on_write, NULL);
    if (rc != 0) {
        MODLOG_DFLT(ERROR, "Error: Failed to write characteristic; rc=%d\n",
                    rc);
        goto err;
    }
    
    return;

err:
    /* Terminate the connection. */
    ble_gap_terminate(peer->conn_handle, BLE_ERR_REM_USER_CONN_TERM);
}

```

## Read Operation

`ble_prox_cent_on_write` reads the tx power level after writing on alert level is completed. If the peer does not support a required service, characteristic, or descriptor OR if a GATT procedure fails , then this function immediately terminates the connection.

```c
static int
ble_prox_cent_on_write(uint16_t conn_handle,
                    const struct ble_gatt_error *error,
                    struct ble_gatt_attr *attr,
                    void *arg)
{   
    MODLOG_DFLT(INFO, "Write alert level char completed; status=%d conn_handle=%d",
                error->status, conn_handle);
    
    /* Read Tx Power level characteristic. */
    const struct peer_chr *chr;
    int rc;
    const struct peer *peer = peer_find(conn_handle);
    
    chr = peer_chr_find_uuid(peer,
                             BLE_UUID16_DECLARE(BLE_SVC_TX_POWER_UUID16),
                             BLE_UUID16_DECLARE(BLE_SVC_PROX_CHR_UUID16_TX_PWR_LVL));
    if (chr == NULL) {
        MODLOG_DFLT(ERROR, "Error: Peer doesn't support the"
                    "Tx power level characteristic\n");
        goto err;
    }
    
    rc = ble_gattc_read(conn_handle, chr->chr.val_handle,
                        ble_prox_cent_on_read, NULL);
    if (rc != 0) {
        MODLOG_DFLT(ERROR, "Error: Failed to read characteristic; rc=%d\n",
                    rc);
        goto err;
    }
    
    return 0;
err:
    /* Terminate the connection. */
    return ble_gap_terminate(peer->conn_handle, BLE_ERR_REM_USER_CONN_TERM);
}

```
## BLE GAP Connect Event

Once the connection is established `BLE_GAP_EVENT_CONNECT` event occurs. Once connected, it proceeds to perform the service discovery. If the connection attempt is failed then central start scanning again using the `ble_prox_cent_scan` function.

```c
case BLE_GAP_EVENT_CONNECT:
        /* A new connection was established or a connection attempt failed. */
        if (event->connect.status == 0) {
            /* Connection successfully established. */
            MODLOG_DFLT(INFO, "Connection established ");

            rc = ble_gap_conn_find(event->connect.conn_handle, &desc);
            assert(rc == 0);
            print_conn_desc(&desc);
            MODLOG_DFLT(INFO, "\n");

            link_supervision_timeout = 6 * desc.conn_itvl;

            /* Remember peer. */
            rc = peer_add(event->connect.conn_handle);
            if (rc != 0) {
                MODLOG_DFLT(ERROR, "Failed to add peer; rc=%d\n", rc);
                return 0;
            }

		< snip >
        } else {
            /* Connection attempt failed; resume scanning. */
            MODLOG_DFLT(ERROR, "Error: Connection failed; status=%d\n",
                        event->connect.status);
            ble_prox_cent_scan();
        }

        return 0;
```

## BLE GAP Disconnect Event

The connection between Central and Peripheral is terminated when the service discovery is failed or the GATT procedure is completed. `ble_gap_terminate` function is used to terminate the connection which results in the event called `BLE_GAP_EVENT_DISCONNECT`.

```c
  case BLE_GAP_EVENT_DISCONNECT:
        /* Connection terminated. */
        MODLOG_DFLT(INFO, "disconnect; reason=%d ", event->disconnect.reason);
        print_conn_desc(&event->disconnect.conn);
        MODLOG_DFLT(INFO, "\n");

        /* Stop calculating path loss, restart once connection is established again */
        conn_peer[event->disconnect.conn.conn_handle].calc_path_loss = false;
        conn_peer[event->disconnect.conn.conn_handle].val_handle = 0;

        /* Forget about peer. */
        peer_delete(event->disconnect.conn.conn_handle);

        ble_prox_cent_scan();
        return 0;
```
## Central Task

`ble_prox_cent_task` is a task which calculate path loss in intervals for all the connected devices. If the path loss exceeds the `high_threshold`, it performs a `ble_gattc_write_no_rsp` to alert prph with path loss to start alerting. If path loss falls below than `low_threshold`, it performs a `ble_gattc_write_no_rsp` to alert prph with path loss to stop alerting.

```c
while (1) {
    for (int i = 0; i <= MYNEWT_VAL(BLE_MAX_CONNECTIONS); i++) {
        if (conn_peer[i].calc_path_loss) {
            MODLOG_DFLT(INFO, "Connection handle : %d", i);
            rc = ble_gap_conn_rssi(i, &rssi);
            if (rc == 0) {
                MODLOG_DFLT(INFO, "Current RSSI = %d", rssi);
            } else {
                MODLOG_DFLT(ERROR, "Failed to get current RSSI");
            }

            path_loss = tx_pwr_lvl - rssi;
            MODLOG_DFLT(INFO, "path loss = %d pwr lvl = %d rssi = %d",
                        path_loss, tx_pwr_lvl, rssi);

            if ((conn_peer[i].val_handle != 0) &&
                    (path_loss > high_threshold || path_loss < low_threshold)) {
                rc = ble_gattc_write_no_rsp_flat(i, conn_peer[i].val_handle,
                                                 &path_loss, sizeof(path_loss));
                if (rc != 0) {
                    MODLOG_DFLT(ERROR, "Error: Failed to write characteristic; rc=%d\n",
                                rc);
                } else {
                    MODLOG_DFLT(INFO, "Write to alert level characteristis done");
                }
            }
        }
    }
    vTaskDelay(500);
}


```

## Threshold

```c
/* Note: Path loss is calculated using formula : threshold - RSSI value
 *       by default threshold is kept -128 as per the spec
 *       high_threshold and low_threshold are hardcoded after testing and noting
 *       RSSI values when distance betweeen devices are less and more.
*/
static int8_t high_threshold = -70;
static int8_t low_threshold = -100;
```

## Conclusion

This Walkthrough covers the code explanation of the BLE_PROX_CENTRAL example. The following points are concluded through this walkthrough.
 1. First a connection is made with a device supporting the Link Loss service.
 2. Gatt operations are performed on the characteristic and results displayed.

