/**
 * SPDX-FileCopyrightText: 2025 Espressif Systems (Shanghai) CO LTD
 *
 *  SPDX-License-Identifier: Apache-2.0
 */
#pragma once

#include <stdint.h>
#ifdef __cplusplus
extern "C" {
#endif

/** Group: Data Memory */

/** Group: Configuration Registers */
/** Type of conf register
 *  ECDSA configuration register
 */
typedef union {
    struct {
        /** work_mode : R/W; bitpos: [1:0]; default: 0;
         *  Configures the working mode of ECDSA accelerator.
         *  0: Signature Verification Mode
         *  1: Signature Generation Mode
         *  2: Public Key Export Mode
         *  3: Invalid
         */
        uint32_t work_mode:2;
        /** ecc_curve : R/W; bitpos: [2]; default: 0;
         *  Configures the elliptic curve used.
         *  0: P-192
         *  1: P-256
         */
        uint32_t ecc_curve:1;
        /** software_set_k : R/W; bitpos: [3]; default: 0;
         *  Configures the generation source of k.
         *  0: k is automatically generated by hardware
         *  1: k is written by software
         */
        uint32_t software_set_k:1;
        /** software_set_z : R/W; bitpos: [4]; default: 0;
         *  Configures how the parameter $z$ is set.
         *  0: Generated from SHA result
         *  1: Written by software
         */
        uint32_t software_set_z:1;
        /** deterministic_k : R/W; bitpos: [5]; default: 0;
         *
         *  Configures how the parameter $k$ is set.
         *  0: Automatically generated by TRNG
         *  1: Generated by the deterministic derivation algorithm
         */
        uint32_t deterministic_k:1;
        uint32_t reserved_6:26;
    };
    uint32_t val;
} ecdsa_conf_reg_t;

/** Type of start register
 *  ECDSA start register
 */
typedef union {
    struct {
        /** start : WT; bitpos: [0]; default: 0;
         *  Configures whether to start the ECDSA operation. This bit will be self-cleared
         *  after configuration.
         *  0: No effect
         *  1: Start the ECDSA operation
         */
        uint32_t start:1;
        /** load_done : WT; bitpos: [1]; default: 0;
         *  Write 1 to generate a signal indicating the ECDSA accelerator's LOAD operation is
         *  done. This bit will be self-cleared after configuration.
         */
        uint32_t load_done:1;
        /** get_done : WT; bitpos: [2]; default: 0;
         *  Write 1 to generate a signal indicating the ECDSA accelerator's GAIN operation is
         *  done. This bit will be self-cleared after configuration.
         */
        uint32_t get_done:1;
        uint32_t reserved_3:29;
    };
    uint32_t val;
} ecdsa_start_reg_t;


/** Group: Clock and Reset Register */
/** Type of clk register
 *  ECDSA clock gate register
 */
typedef union {
    struct {
        /** clk_gate_force_on : R/W; bitpos: [0]; default: 0;
         *  Configures whether to force on ECDSA memory clock gate.
         *  0: No effect
         *  1: Force on
         */
        uint32_t clk_gate_force_on:1;
        uint32_t reserved_1:31;
    };
    uint32_t val;
} ecdsa_clk_reg_t;


/** Group: Interrupt Registers */
/** Type of int_raw register
 *  ECDSA interrupt raw register
 */
typedef union {
    struct {
        /** prep_done_int_raw : R/SS/WTC; bitpos: [0]; default: 0;
         *  The raw interrupt status of the ECDSA_PREP_DONE_INT interrupt.
         */
        uint32_t prep_done_int_raw:1;
        /** proc_done_int_raw : R/SS/WTC; bitpos: [1]; default: 0;
         *  The raw interrupt status of the ECDSA_PROC_DONE_INT interrupt.
         */
        uint32_t proc_done_int_raw:1;
        /** post_done_int_raw : R/SS/WTC; bitpos: [2]; default: 0;
         *  The raw interrupt status of the ECDSA_POST_DONE_INT interrupt.
         */
        uint32_t post_done_int_raw:1;
        /** sha_release_int_raw : R/SS/WTC; bitpos: [3]; default: 0;
         *  The raw interrupt status of the ECDSA_SHA_RELEASE_INT interrupt.
         */
        uint32_t sha_release_int_raw:1;
        uint32_t reserved_4:28;
    };
    uint32_t val;
} ecdsa_int_raw_reg_t;

/** Type of int_st register
 *  ECDSA interrupt status register
 */
typedef union {
    struct {
        /** prep_done_int_st : RO; bitpos: [0]; default: 0;
         *  The masked interrupt status of the ECDSA_PREP_DONE_INT interrupt.
         */
        uint32_t prep_done_int_st:1;
        /** proc_done_int_st : RO; bitpos: [1]; default: 0;
         *  The masked interrupt status of the ECDSA_PROC_DONE_INT interrupt.
         */
        uint32_t proc_done_int_st:1;
        /** post_done_int_st : RO; bitpos: [2]; default: 0;
         *  The masked interrupt status of the ECDSA_POST_DONE_INT interrupt.
         */
        uint32_t post_done_int_st:1;
        /** sha_release_int_st : RO; bitpos: [3]; default: 0;
         *  The masked interrupt status of the ECDSA_SHA_RELEASE_INT interrupt.
         */
        uint32_t sha_release_int_st:1;
        uint32_t reserved_4:28;
    };
    uint32_t val;
} ecdsa_int_st_reg_t;

/** Type of int_ena register
 *  ECDSA interrupt enable register
 */
typedef union {
    struct {
        /** prep_done_int_ena : R/W; bitpos: [0]; default: 0;
         *  Write 1 to enable the ECDSA_PREP_DONE_INT interrupt.
         */
        uint32_t prep_done_int_ena:1;
        /** proc_done_int_ena : R/W; bitpos: [1]; default: 0;
         *  Write 1 to enable the ECDSA_PROC_DONE_INT interrupt.
         */
        uint32_t proc_done_int_ena:1;
        /** post_done_int_ena : R/W; bitpos: [2]; default: 0;
         *  Write 1 to enable the ECDSA_POST_DONE_INT interrupt.
         */
        uint32_t post_done_int_ena:1;
        /** sha_release_int_ena : R/W; bitpos: [3]; default: 0;
         *  Write 1 to enable the ECDSA_SHA_RELEASE_INT interrupt.
         */
        uint32_t sha_release_int_ena:1;
        uint32_t reserved_4:28;
    };
    uint32_t val;
} ecdsa_int_ena_reg_t;

/** Type of int_clr register
 *  ECDSA interrupt clear register
 */
typedef union {
    struct {
        /** prep_done_int_clr : WT; bitpos: [0]; default: 0;
         *  Write 1 to clear the ECDSA_PREP_DONE_INT interrupt.
         */
        uint32_t prep_done_int_clr:1;
        /** proc_done_int_clr : WT; bitpos: [1]; default: 0;
         *  Write 1 to clear the ECDSA_PROC_DONE_INT interrupt.
         */
        uint32_t proc_done_int_clr:1;
        /** post_done_int_clr : WT; bitpos: [2]; default: 0;
         *  Write 1 to clear the ECDSA_POST_DONE_INT interrupt.
         */
        uint32_t post_done_int_clr:1;
        /** sha_release_int_clr : WT; bitpos: [3]; default: 0;
         *  Write 1 to clear the ECDSA_SHA_RELEASE_INT interrupt.
         */
        uint32_t sha_release_int_clr:1;
        uint32_t reserved_4:28;
    };
    uint32_t val;
} ecdsa_int_clr_reg_t;


/** Group: Status Registers */
/** Type of state register
 *  ECDSA status register
 */
typedef union {
    struct {
        /** busy : RO; bitpos: [1:0]; default: 0;
         *  Represents the working status of the ECDSA accelerator.
         *  0: IDLE
         *  1: LOAD
         *  2: GAIN
         *  3: BUSY
         */
        uint32_t busy:2;
        uint32_t reserved_2:30;
    };
    uint32_t val;
} ecdsa_state_reg_t;


/** Group: Result Register */
/** Type of result register
 *  ECDSA result register
 */
typedef union {
    struct {
        /** operation_result : RO/SS; bitpos: [0]; default: 0;
         *  Indicates if the ECDSA operation is successful.
         *  0: Not successful
         *  1: Successful
         *  Only valid when the ECDSA operation is done.
         */
        uint32_t operation_result:1;
        uint32_t reserved_1:31;
    };
    uint32_t val;
} ecdsa_result_reg_t;


/** Group: SHA Registers */
/** Type of sha_mode register
 *  ECDSA SHA-control register (Hash algrithm)
 */
typedef union {
    struct {
        /** sha_mode : R/W; bitpos: [2:0]; default: 0;
         *  Configures SHA algorithms for message hash.
         *  1: SHA-224
         *  2: SHA-256
         *  Others: invalid
         */
        uint32_t sha_mode:3;
        uint32_t reserved_3:29;
    };
    uint32_t val;
} ecdsa_sha_mode_reg_t;

/** Type of sha_start register
 *  ECDSA SHA-control register (operation)
 */
typedef union {
    struct {
        /** sha_start : WT; bitpos: [0]; default: 0;
         *  Write 1 to start the first SHA operation in the ECDSA process. This bit will be
         *  self-cleared after configuration.
         */
        uint32_t sha_start:1;
        uint32_t reserved_1:31;
    };
    uint32_t val;
} ecdsa_sha_start_reg_t;

/** Type of sha_continue register
 *  ECDSA SHA-control register (operation)
 */
typedef union {
    struct {
        /** sha_continue : WT; bitpos: [0]; default: 0;
         *  Write 1 to start the latter SHA operation in the ECDSA process. This bit will be
         *  self-cleared after configuration.
         */
        uint32_t sha_continue:1;
        uint32_t reserved_1:31;
    };
    uint32_t val;
} ecdsa_sha_continue_reg_t;

/** Type of sha_busy register
 *  ECDSA SHA-control status register
 */
typedef union {
    struct {
        /** sha_busy : RO; bitpos: [0]; default: 0;
         *  Represents the working status of the SHA accelerator in the ECDSA process.
         *  0: IDLE
         *  1: BUSY
         */
        uint32_t sha_busy:1;
        uint32_t reserved_1:31;
    };
    uint32_t val;
} ecdsa_sha_busy_reg_t;


/** Group: Version Register */
/** Type of date register
 *  Version control register
 */
typedef union {
    struct {
        /** date : R/W; bitpos: [27:0]; default: 37761312;
         *  The ECDSA version control register.
         */
        uint32_t date:28;
        uint32_t reserved_28:4;
    };
    uint32_t val;
} ecdsa_date_reg_t;


typedef struct {
    uint32_t reserved_000;
    volatile ecdsa_conf_reg_t conf;
    volatile ecdsa_clk_reg_t clk;
    volatile ecdsa_int_raw_reg_t int_raw;
    volatile ecdsa_int_st_reg_t int_st;
    volatile ecdsa_int_ena_reg_t int_ena;
    volatile ecdsa_int_clr_reg_t int_clr;
    volatile ecdsa_start_reg_t start;
    volatile ecdsa_state_reg_t state;
    volatile ecdsa_result_reg_t result;
    uint32_t reserved_028[53];
    volatile ecdsa_date_reg_t date;
    uint32_t reserved_100[64];
    volatile ecdsa_sha_mode_reg_t sha_mode;
    uint32_t reserved_204[3];
    volatile ecdsa_sha_start_reg_t sha_start;
    volatile ecdsa_sha_continue_reg_t sha_continue;
    volatile ecdsa_sha_busy_reg_t sha_busy;
    uint32_t reserved_21c[25];
    volatile uint32_t message[8];
    uint32_t reserved_2a0[40];
    volatile uint32_t r[8];
    volatile uint32_t s[8];
    volatile uint32_t z[8];
    volatile uint32_t qax[8];
    volatile uint32_t qay[8];
} ecdsa_dev_t;

extern ecdsa_dev_t ECDSA;

#ifndef __cplusplus
_Static_assert(sizeof(ecdsa_dev_t) == 0x3e0, "Invalid size of ecdsa_dev_t structure");
#endif

#ifdef __cplusplus
}
#endif
