/*
 * SPDX-FileCopyrightText: 2018-2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#pragma once

#include <esp_err.h>
#include "esp_event.h"

#ifdef __cplusplus
extern "C" {
#endif

ESP_EVENT_DECLARE_BASE(PROTOCOMM_SECURITY_SESSION_EVENT);

/**
 * @brief   Events generated by the protocomm security layer
 *
 * These events are generated while establishing secured session.
 */
typedef enum {
    PROTOCOMM_SECURITY_SESSION_SETUP_OK,                 /**< Secured session established successfully */
    PROTOCOMM_SECURITY_SESSION_INVALID_SECURITY_PARAMS,  /**< Received invalid (NULL) security parameters (username / client public-key) */
    PROTOCOMM_SECURITY_SESSION_CREDENTIALS_MISMATCH,     /**< Received incorrect credentials (username / PoP) */
} protocomm_security_session_event_t;

/**
 * @brief   Protocomm Security 1 parameters: Proof Of Possession
 */
typedef struct protocomm_security1_params {
    /**
     * Pointer to buffer containing the proof of possession data
     */
    const uint8_t *data;

    /**
     * Length (in bytes) of the proof of possession data
     */
    uint16_t len;
} protocomm_security1_params_t;


/**
 * @brief Protocomm Security 2 parameters: Salt and Verifier
 *
 */
typedef struct protocomm_security2_params {
    /**
     * Pointer to the buffer containing the salt
     */
    const char *salt;

    /**
     * Length (in bytes) of the salt
     */
    uint16_t salt_len;

    /**
     * Pointer to the buffer containing the verifier
     */
    const char *verifier;

    /**
     * Length (in bytes) of the verifier
     */
    uint16_t verifier_len;
} protocomm_security2_params_t;

typedef void * protocomm_security_handle_t;

/**
 * @brief   Protocomm security object structure.
 *
 * The member functions are used for implementing secure
 * protocomm sessions.
 *
 * @note    This structure should not have any dynamic
 *          members to allow re-entrancy
 */
typedef struct protocomm_security {
    /**
     * Unique version number of security implementation
     */
    int ver;

    /**
     * Patch version number of security implementation
     */
    uint8_t patch_ver;

    /**
     * Function for initializing/allocating security
     * infrastructure
     */
    esp_err_t (*init)(protocomm_security_handle_t *handle);

    /**
     * Function for deallocating security infrastructure
     */
    esp_err_t (*cleanup)(protocomm_security_handle_t handle);

    /**
     * Starts new secure transport session with specified ID
     */
    esp_err_t (*new_transport_session)(protocomm_security_handle_t handle,
                                       uint32_t session_id);

    /**
     * Closes a secure transport session with specified ID
     */
    esp_err_t (*close_transport_session)(protocomm_security_handle_t handle,
                                         uint32_t session_id);

    /**
     * Handler function for authenticating connection
     * request and establishing secure session
     */
    esp_err_t (*security_req_handler)(protocomm_security_handle_t handle,
                                      const void *sec_params,
                                      uint32_t session_id,
                                      const uint8_t *inbuf, ssize_t inlen,
                                      uint8_t **outbuf, ssize_t *outlen,
                                      void *priv_data);

    /**
     * Function which implements the encryption algorithm
     */
    esp_err_t (*encrypt)(protocomm_security_handle_t handle,
                         uint32_t session_id,
                         const uint8_t *inbuf, ssize_t inlen,
                         uint8_t **outbuf, ssize_t *outlen);

    /**
     * Function which implements the decryption algorithm
     */
    esp_err_t (*decrypt)(protocomm_security_handle_t handle,
                         uint32_t session_id,
                         const uint8_t *inbuf, ssize_t inlen,
                         uint8_t **outbuf, ssize_t *outlen);
} protocomm_security_t;

#ifdef __cplusplus
}
#endif
